(*************************************************************************
 *  ShadesInOutU.pas                                                     *
 *  Vladimr Slvik 2007-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Additional PNG management routines for Shades :                      *
 *    chunk list management etc.                                         *
 *                                                                       *
 *  -additional libraries: PNGImage                                      *
 *************************************************************************)

unit ShadesInOutU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------
interface

uses PNGImage, Classes, GR32;
//------------------------------------------------------------------------------

function FileIsPNG(const FileName: String): Boolean;
// is a PNG file or not

procedure SaveChunks(const PNG: TPNGObject; const List: TList);
procedure LoadChunks(const PNG: TPNGObject; const List: TList);
// save and load safe-to-keep chunks
// -> last char/byte is lowercase ascii

procedure ClearChunkList(const List: TList);
// or kill them all? :)

procedure AddTimestamp(const PNG: TPNGObject);
// what it says...

procedure DumpWithAlpha(const Bitmap: TBitmap32; const FileName: String);
// for debugging

//==============================================================================
implementation

uses SysUtils, DateUtils,
     ConstStrU;

//------------------------------------------------------------------------------

function FileIsPNG(const FileName: String): Boolean;
type TPNGSig = array [0 .. 7] of Byte;
var F: TFileStream;
    A: TPNGSig;
    i: Integer;
const ValidSig: TPNGSig = (137, 80, 78, 71, 13, 10, 26, 10);
begin
  Result:= False;
  try try
    F:= nil;
    if not FileExists(FileName) then Abort;
    F:= TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
    if F.Size < 58 then Abort;
    (* minimum 58 bytes, because:
       8               =   8  magic header
       4 + 4 + 13 + 4  =  25  IHDR
       4 + 4 +  1 + 4  =  13  IDAT (nonempty!)
       4 + 4 +  0 + 4  =  12  IEND
                         ----
                          58
    *)
    F.Read(A, Sizeof(A));
    for i:= Low(A) to High(A) do if A[i] <> ValidSig[i] then Abort;
    Result:= True;
  except end finally // duh ... want it caught and eaten properly inside, do this
    F.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure SaveChunks(const PNG: TPNGObject; const List: TList);
// backup chunk data to list
var i: Integer;
    C: TChunk;
    S: TStream;
begin
  with PNG.Chunks do if Count > 0 then for i:= 0 to Count - 1 do begin
    C:= Item[i];
    if C.Name[4] in ['a'..'z'] then begin
      S:= TMemoryStream.Create;
      C.SaveToStream(S);
      List.Add(S);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure LoadChunks(const PNG: TPNGObject; const List: TList);
// copy chunk data from list to PNG object
// all chunks are safe before first IDAT, except for their own ordering
// upstream code ensures that only harmless chunks are copied over
// -> no qualms about inserting them as they come
var i: Integer;
    C: TChunk;
    S: TStream;
    IdatPos: Integer;
    Name: TChunkName;
    DataSize: Cardinal;
begin
  C:= PNG.Chunks.FindChunk(TChunkIDAT);
  IdatPos:= C.Index;
  // first find where IDAT is
  if IdatPos = -1 then
    raise EPNGNoImageData.Create('ShadesInOutU.LoadChunks: Could not find IDAT.');
  // die if it's not there (?)
  with List do if Count > 0 then for i:= 0 to Count - 1 do begin
    C:= TChunk.Create(PNG);
    PNG.Chunks.Insert(C, IdatPos); // my PNGImage is hacked, this is private property!
    S:= TMemoryStream(Items[i]);
    S.Position:= 0;
    S.Read(DataSize, 4);
    DataSize:= ByteSwap(DataSize);
    S.Read(Name, 4);
    C.LoadFromStream(S, Name, DataSize);
  end;
end;

//------------------------------------------------------------------------------

procedure ClearChunkList(const List: TList);
var i: Integer;
    S: TMemoryStream;
begin
  with List do if Count > 0 then for i:= Count - 1 downto 0 do begin
    S:= TMemoryStream(Items[i]);
    S.Free;
    Delete(i);
  end;
end;

//------------------------------------------------------------------------------

procedure AddTimestamp(const PNG: TPNGObject);
var Time: TChunktIME;
    TimeSrc: TDateTime;
begin
  Time:= PNG.Chunks.FindChunk(TChunktIME) as TChunktIME;
  if not Assigned(Time) then Time:= PNG.Chunks.Add(TChunktIME) as TChunktIME;
  TimeSrc:= Now;
  with Time do begin
    Year:= YearOf(TimeSrc);
    Month:= MonthOf(TimeSrc);
    Day:= DayOf(TimeSrc);
    Hour:= HourOf(TimeSrc);
    Minute:= MinuteOf(TimeSrc);
    Second:= SecondOf(TimeSrc);
  end;
end;

//------------------------------------------------------------------------------

procedure DumpWithAlpha(const Bitmap: TBitmap32; const FileName: String);
var AlphaChannel: TBitmap32;
    i: Integer;
    C: TColor32;
begin
  Bitmap.SaveToFile(FileName);
  AlphaChannel:= TBitmap32.Create;
  AlphaChannel.SetSizeFrom(Bitmap);
  for i:= 0 to Bitmap.Width * Bitmap.Height - 1 do begin
    C:= AlphaComponent(Bitmap.Bits[i]);
    C:= $FF000000 or C or C shl 8 or C shl 16;
    AlphaChannel.Bits[i]:= C;
  end;
  AlphaChannel.SaveToFile(FileName + '-alpha.bmp');
  AlphaChannel.Free;
end;

//------------------------------------------------------------------------------

end.
