(*************************************************************************
 *  BrushButton.pas                                                      *
 *  Vladimr Slvik 2010                                                 *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  component that ecapsulates all logic needed for selecting a brush:   *
 *    Not really complete, should be used only as dynamically created.   *
 *    Also the external dependency on resource makes it a bit unclean.   *
 *                                                                       *
 *  -additional libraries: GR32, PNGImage                                *
 *************************************************************************)

unit BrushButton;

{$INCLUDE ..\Switches.inc}
{t default -}

//==============================================================================

interface

uses SysUtils, Windows, Classes, Graphics, Controls, StdCtrls, Buttons, Menus, GR32;

//------------------------------------------------------------------------------

type
  TBrushButton = class(TBitBtn)
  private
    FImages: TImageList;
    FMenu: TPopupMenu;
    FDefaultBrush: Integer;
    FBrushSize: Integer;
    FBrushCount: Integer;
    FMenuRowCount: Integer;
    FBrushImages: TBitmap32;
    FCurrentBrushIndex: Integer;
    FNewBrushIndex: Integer;
    FCurrentBrush: TBitmap32;
    FOnBrushSelect: TNotifyEvent;
    procedure UpdateCurrentBrush;
    procedure UpdateGlyph;
    procedure LoadResources;
    procedure CreateMenu;
    procedure AddItem(const AIndex: Integer);
    procedure MenuClick(Sender: TObject);
    procedure MeasureItem(Sender: TObject; ACanvas: TCanvas; var Width, Height: Integer);
  protected // hiding properties
    property Height;
    property Width;
    property Glyph;
    property Layout;
    property Kind;
    property Margin;
    property NumGlyphs;
    property Spacing;
    property Style;
    property Cancel;
    property Default;
    property ModalResult;
    property Caption;
    property OnContextPopup;
    property PopupMenu;
    property Font;
    property Brush;
    property ParentFont;
    property WordWrap;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AfterConstruction; override;
    procedure Click; override;
  published
    property SelectedBrush: TBitmap32 read FCurrentBrush;
    property OnBrushSelect: TNotifyEvent read FOnBrushSelect write FOnBrushSelect;
  end;

//==============================================================================

implementation

uses Math, Dialogs, PngImage;

//==============================================================================

constructor TBrushButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBrushImages:= TBitmap32.Create;
  FCurrentBrush:= TBitmap32.Create;
  FDefaultBrush:= 0;
  FBrushSize:= 24;
  FBrushCount:= 0;
  FMenuRowCount:= 8;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.AfterConstruction;
begin
  inherited AfterConstruction;
  FImages:= TImageList.Create(Self);
  FMenu:= TPopupMenu.Create(Self);
  FMenu.Images:= FImages;
  FMenu.OwnerDraw:= True;
  LoadResources;
  CreateMenu;
  UpdateCurrentBrush;
  UpdateGlyph;
  Height:= FBrushSize + 8;
  Width:= FBrushSize + 8;
end;

//------------------------------------------------------------------------------

destructor TBrushButton.Destroy;
begin
  FBrushImages.Free;
  FCurrentBrush.Free;
  FImages.Free;
  FMenu.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.UpdateCurrentBrush;
var Src: TRect;
begin
  FCurrentBrushIndex:= EnsureRange(FCurrentBrushIndex, 0, FBrushImages.Width div FBrushSize);
  FCurrentBrush.SetSize(FBrushSize, FBrushSize);
  with Src do begin
    Top:= 0;
    Bottom:= FBrushSize;
    Left:= FCurrentBrushIndex * FBrushSize;
    Right:= Left + FBrushSize;
  end;
  FBrushImages.DrawTo(FCurrentBrush, 0, 0, Src);
end;

//------------------------------------------------------------------------------

procedure TBrushButton.AddItem(const AIndex: Integer);
var NewItem: TMenuItem;
begin
  NewItem:= TMenuItem.Create(FMenu);
  with NewItem do begin
    Tag:= AIndex;
    OnClick:= MenuClick;
    Name:= 'DynMenuItem' + IntToStr(AIndex);
    Caption:= '';
    Visible:= True;
    ImageIndex:= AIndex; //FBrushCount - AIndex - 1;
    OnMeasureItem:= MeasureItem;
    if ((AIndex - 1) mod FMenuRowCount = (FMenuRowCount - 1)) then Break:= mbBarBreak;  
  end;
  FMenu.Items.Add(NewItem);
end;

//------------------------------------------------------------------------------

procedure TBrushButton.LoadResources;
var RR: TResourceStream;
    PR: TPNGObject;
begin
  try
    RR:= TResourceStream.Create(HInstance, 'BrushData', RT_RCDATA);
    RR.Read(FDefaultBrush, 4);
    RR.Read(FBrushSize, 4);
    RR.Read(FMenuRowCount, 4);
  finally
    RR.Free;
  end;
  FCurrentBrushIndex:= FDefaultBrush;
  try
    PR:= TPNGObject.Create;
    PR.LoadFromResourceName(hInstance, 'Brushes');
    FBrushImages.Assign(PR);
  finally
    PR.Free;
  end;
  FBrushCount:= FBrushImages.Width div FBrushSize;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.CreateMenu;
var i: Integer;
    LoaderBitmap: TBitmap;
begin
  Assert(FBrushImages.Width > 0);
  LoaderBitmap:= TBitmap.Create;
  UpdateCurrentBrush;
  LoaderBitmap.Height:= FBrushSize;
  LoaderBitmap.Width:= FBrushSize * FBrushCount;
  LoaderBitmap.TransparentColor:= clWhite;
  FBrushImages.DrawTo(LoaderBitmap.Canvas.Handle, 0, 0);
  FImages.Height:= FBrushSize;
  FImages.Width:= FBrushSize;
  Fimages.AddMasked(LoaderBitmap, clWhite);
  for i:= 0 to FBrushCount - 1 do
    Additem(i);
  LoaderBitmap.Free;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.MenuClick(Sender: TObject);
begin
  FNewBrushIndex:= (Sender as TComponent).Tag;
  if FNewBrushIndex <> FCurrentBrushIndex then begin
    FCurrentBrushIndex:= FNewBrushIndex;
    UpdateCurrentBrush;
    UpdateGlyph;
    if Assigned(FOnBrushSelect) then FOnBrushSelect(Self);
  end;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.Click;
var P: TPoint;
begin
  P:= ClientToScreen(Point(0, Height));
  FMenu.Popup(P.X, P.Y);
  inherited Click;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.UpdateGlyph;
var B: TBitmap;
begin
  B:= TBitmap.Create;
  FImages.GetBitmap(FCurrentBrushIndex, B);
  Glyph.Assign(B);
  B.Free;
end;

//------------------------------------------------------------------------------

procedure TBrushButton.MeasureItem(Sender: TObject; ACanvas: TCanvas;
  var Width, Height: Integer);
begin
  Width:= FBrushSize - 10;
  Height:= FBrushSize + 2;
  // why 10 and 2, only God knows - but then it's brush + 1 px frame
end;

//------------------------------------------------------------------------------

end.
